<?php
namespace AIOSEO\Plugin\Addon\LinkAssistant\Api;

use AIOSEO\Plugin\Addon\LinkAssistant\Models;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles all endpoints for the Links Report.
 *
 * @since 1.0.0
 */
class LinksReport {
	/**
	 * The posts limit.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	private static $postsLimit = 10;

	/**
	 * Returns paginated domain results.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function paginate( $request ) {
		$filter = $request['filter'];
		$termId = $request['termId'];

		$searchTerm = ! empty( $request['searchTerm'] ) ? urldecode( $request['searchTerm'] ) : '';

		$page   = (int) $request['page'];
		$offset = 1 === $page ? 0 : ( $page - 1 ) * self::$postsLimit;

		$supportedFilters = [
			'all',
			'linking-opportunities',
			'orphaned-posts'
		];

		$supportedFilters = array_merge( $supportedFilters, aioseoLinkAssistant()->helpers->getIncludedPostTypes() );

		if ( $filter && ! in_array( $filter, $supportedFilters, true ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid filter was passed.'
			], 404 );
		}

		return new \WP_REST_Response( [
			'success'     => true,
			'linksReport' => aioseoLinkAssistant()->helpers->getLinksReportData( $offset, $searchTerm, $filter, $termId )
		], 200 );
	}

	/**
	 * Search for a specific domain.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function search( $request ) {
		$searchTerm = urldecode( $request['searchTerm'] );

		return new \WP_REST_Response( [
			'success'     => true,
			'linksReport' => aioseoLinkAssistant()->helpers->getLinksReportData( 0, $searchTerm )
		], 200 );
	}

	/**
	 * Returns filtered domain results.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function filter( $request ) {
		$filter = $request['filter'];
		$termId = $request['termId'];

		$supportedFilters = [
			'all',
			'linking-opportunities',
			'orphaned-posts'
		];

		$supportedFilters = array_merge( $supportedFilters, aioseoLinkAssistant()->helpers->getIncludedPostTypes() );

		if ( ! in_array( $filter, $supportedFilters, true ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid filter was passed.'
			], 404 );
		}

		return new \WP_REST_Response( [
			'success'     => true,
			'linksReport' => aioseoLinkAssistant()->helpers->getLinksReportData( 0, '', $filter, $termId )
		], 200 );
	}

	/**
	 * Deletes all links for the given post ID.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function deletePostLinks( $request ) {
		$postId = $request['postId'];

		$filter = ! empty( $request['filter'] ) ? urldecode( $request['filter'] ) : '';
		$termId = ! empty( $request['termId'] ) ? urldecode( $request['termId'] ) : '';

		$searchTerm = ! empty( $request['searchTerm'] ) ? urldecode( $request['searchTerm'] ) : '';

		$page   = ! empty( $request['currentPage'] ) ? (int) $request['currentPage'] : 1;
		$offset = 1 === $page ? 0 : ( $page - 1 ) * self::$postsLimit;

		if ( ! $postId ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid post ID was passed.'
			], 404 );
		}

		$links   = Models\Link::getLinks( $postId );
		$linkIds = array_map( function( $link ) {
			return $link->id;
		}, $links );

		aioseoLinkAssistant()->helpers->deleteLinksInPost( $linkIds );

		return new \WP_REST_Response( [
			'success'     => true,
			'linksReport' => aioseoLinkAssistant()->helpers->getLinksReportData( $offset, $searchTerm, $filter, $termId )
		], 200 );
	}
}