<?php
namespace AIOSEO\Plugin\Addon\LinkAssistant\Api;

use AIOSEO\Plugin\Addon\LinkAssistant\Models;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles all endpoints for the Domains Report.
 *
 * @since 1.0.0
 */
class DomainsReport {
	/**
	 * The domains limit.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	private static $domainsLimit = 10;

	/**
	 * Returns paginated domain results.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function paginate( $request ) {
		$page       = (int) $request['page'];
		$searchTerm = urldecode( $request['searchTerm'] );
		$offset     = 1 === $page ? 0 : ( $page - 1 ) * self::$domainsLimit;

		return new \WP_REST_Response( [
			'success'       => true,
			'domainsReport' => aioseoLinkAssistant()->helpers->getDomainsReportData( $offset, $searchTerm )
		], 200 );
	}

	/**
	 * Search for a specific domain.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function search( $request ) {
		$page       = (int) $request['page'];
		$offset     = 1 === $page ? 0 : ( $page - 1 ) * self::$domainsLimit;
		$searchTerm = ! empty( $request['searchTerm'] ) ? urldecode( $request['searchTerm'] ) : '';

		return new \WP_REST_Response( [
			'success'       => true,
			'domainsReport' => aioseoLinkAssistant()->helpers->getDomainsReportData( $offset, $searchTerm )
		], 200 );
	}

	/**
	 * Processes a bulk action on domains.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function bulk( $request ) {
		$body        = $request->get_json_params();
		$currentPage = ! empty( $body['currentPage'] ) ? (int) $body['currentPage'] : 1;
		$searchTerm  = ! empty( $body['searchTerm'] ) ? $body['searchTerm'] : '';
		$offset      = 1 === $currentPage ? 0 : ( $currentPage - 1 ) * self::$domainsLimit;
		$hostnames   = ! empty( $body['hostnames'] ) ? $body['hostnames'] : [];

		if ( empty( $hostnames ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid domains were passed.'
			], 400 );
		}

		foreach ( $hostnames as $hostname ) {
			$links   = Models\Link::getDomainLinks( $hostname );
			$linkIds = array_map( function( $link ) {
				return $link->id;
			}, $links );

			aioseoLinkAssistant()->helpers->deleteLinksInPost( $linkIds );
		}

		return new \WP_REST_Response( [
			'success'       => true,
			'domainsReport' => aioseoLinkAssistant()->helpers->getDomainsReportData( $offset, $searchTerm )
		], 200 );
	}
}